//------------------------------------------------------------------------------
// File: WhiteWong.cs
// Controls the White Trail/Wong Trail task that forces the player which 
// character to side with
// Author: Matthew Rudge
//------------------------------------------------------------------------------

$WhiteWong::TaskName          = "WhiteWongTask";
$WhiteWong::LogMainTask       = "White Way, Wong Way";
$WhiteWong::LogMainTaskDesc   = "Track down White and Wong and bring them to justice.";
$WhiteWong::LogMainTaskReward = "Reward: Capturing White and Wong.";
$WhiteWong::LogFireTask       = "Hot Times";
$WhiteWong::LogFireTaskDesc   = "Put out the Fire in the little settlement along White Way (or Wong Way).";
$WhiteWong::LogFireTaskReward = "Reward: A nice base of operations.";
$WhiteWong::LogFindBrief      = "Find White and Wong.";
$WhiteWong::LogFindLong       = "Find Barnabas White and Wong Fu.";
$WhiteWong::LogFireBrief      = "Help put out the Fire.";
$WhiteWong::LogFireLong       = "Help the townsfolk put out the fire that White and Wong started.";
$WhiteWong::LogChooseBrief    = "Capture White or Wong.";
$WhiteWong::LogChooseLong     = "You'll only be able to capture White or Wong, not both. Make your choice carefully.";

$WhiteWong::DynamiteDamage = 20;


/******************************************************************************/
/** WHITE WONG TASK HELPERS
/******************************************************************************/
////////////////////////////////////////////////////////////////////////////////
//! Activates the white or wong tasks
////////////////////////////////////////////////////////////////////////////////
function startWhiteWongTasks()
{
   // Start main quest
   InitializeQuestGui();
   tsAddQuestToTracker(
      true, 
      $WhiteWong::LogMainTask, 
      "", 
      $WhiteWong::LogMainTaskDesc, 
      $WhiteWong::LogMainTaskReward
      );
   tsReadyQuest($WhiteWong::LogMainTask);
   tsAddTaskToTracker($WhiteWong::LogMainTask, $WhiteWong::LogFindBrief, $WhiteWong::LogFindBrief);
   
   tsStartTask($WhiteWong::TaskName, tsGetStartingVertexes($WhiteWong::TaskName));
   tsEnableCommand("BuildCmdData", false);
   
   tsConvertObjTeam("whitebandit1", $OST_NEUTRAL);
   tsConvertObjTeam("whitebandit2", $OST_NEUTRAL);
   tsConvertObjTeam("whitebandit3", $OST_NEUTRAL);
   tsConvertObjTeam("whitebandit4", $OST_NEUTRAL);
   tsConvertObjTeam("wongbandit1", $OST_NEUTRAL);
   tsConvertObjTeam("wongbandit2", $OST_NEUTRAL);
   tsConvertObjTeam("wongbandit3", $OST_NEUTRAL);
   tsConvertObjTeam("wongbandit4", $OST_NEUTRAL);
}
////////////////////////////////////////////////////////////////////////////////
//! Deactivates white or wong tasks
////////////////////////////////////////////////////////////////////////////////
function endWhiteWongTasks()
{
   tsEndTask($WhiteWong::TaskName);
   tsEnableCommand("BuildCmdData", true);
}
////////////////////////////////////////////////////////////////////////////////
//! Called if white has been chosen
////////////////////////////////////////////////////////////////////////////////
function chooseWhite()
{
   // Spawn white on main map
   %pos = HeroCharacter.getPosition();
   %obj = tsCreateObj("MrWhiteData", "mrwhite", $OST_PLAYER, %pos, "mrwhite");
   %obj.name = "Mr.White ";
   
   Bigphony.deleteObject();
   tsCompleteVertexes("WhiteWongChoice", "WhiteChosen");
   startPrepForMadRussian();
   // Reward   
   tsModifyResource($Resource::Experience, 25, true);
   SendProductionToClient(HeroCharacter, "25 experience");
}
////////////////////////////////////////////////////////////////////////////////
//! Called if wong has been chosen
////////////////////////////////////////////////////////////////////////////////
function chooseWong()
{
   // Spawn wong on main map
   %pos = HeroCharacter.getPosition();
   %obj = tsCreateObj("MrWongData", "mrwong", $OST_PLAYER, %pos, "mrwong");
   %obj.name = "Mr.Wong ";
   
   Bigphony.deleteObject();
   tsCompleteVertexes("WhiteWongChoice", "WongChosen");
   startPrepForMadRussian();
   
   // Reward   
   tsModifyResource($Resource::Experience, 25, true);
   SendProductionToClient(HeroCharacter, "25 experience");
}
////////////////////////////////////////////////////////////////////////////////
//! Creates and returns a default fire disaster
////////////////////////////////////////////////////////////////////////////////
function whiteWongCreateFire()
{
   // Set up the fire disaster that will be used
   %fireDisaster = new SLFireScenario() {
      chance = 100;
      duration = -1;
      
      startStrength = 1;
      
      spreadCount = 2;
      spreadTime = 6;
      spreadChance = 30;
      spreadRadius = 25;
      spreadStrength = 4;
      
      burnTime = 10;
      burnStrength = 1;
      burnMax = 2;
      
      // does no damage
      damageTime = 0.1;
      damageStrength = 0.2;
   };
   return %fireDisaster;
}
////////////////////////////////////////////////////////////////////////////////
//! Sets a building ablaze by adding it to the fire disaster
//! \param %obj Object with building component
//! \param %fire Fire disaster
////////////////////////////////////////////////////////////////////////////////
function setBuildingAblaze(%obj, %fire)
{
   if(!isObject(%obj) || !isObject(%fire)) {
      return;
   }
   
   %building = slgQueryInterface(%obj, $CID_BUILDING);
   if(!isObject(%building)) {
      return;
   }
   
   // Catch fire if not already on fire
   %building.startFire(%fire);
}
function whiteWongTaskPrisoner()
{
   if(WhiteWongChooseWhite.hasBeenCompleted()) {
      return "mrwhite";
   }
   else {
      return "mrwong";
   }
}
function whiteWongTaskEscapee()
{
   if(WhiteWongChooseWhite.hasBeenCompleted()) {
      return "mrwong";
   }
   else {
      return "mrwhite";
   }
}


/******************************************************************************/
/** WHITE WONG TASK
/******************************************************************************/
//-WhiteWongTownDialog Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Gives intro dialog between hero and townsfolk
////////////////////////////////////////////////////////////////////////////////
function WhiteWongTownDialog::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Send any intro dialog, ending with dialog with dialog id
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_introone", $TaskButton::Okay, "HeroCharacter");
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_introend", $TaskButton::Okay, "Civ1", "id_wow_introend");
}
//-WhiteWongFireTown Vertex-----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Responsible for converting the town to the player's team and then setting it
// ablaze mwahaha
////////////////////////////////////////////////////////////////////////////////
function WhiteWongFireTown::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Convert teams
   tsConvertObjTeam("WWwell1", $OST_PLAYER);
   tsConvertObjTeam("WWwell2", $OST_PLAYER);
   tsConvertObjTeam("WWcabin1", $OST_PLAYER);
   tsConvertObjTeam("WWcabin2", $OST_PLAYER);
   tsConvertObjTeam("WWblacksmith", $OST_PLAYER);
   tsConvertObjTeam("WWsaloon", $OST_PLAYER);
   tsConvertObjTeam("WWstore", $OST_PLAYER);
   tsConvertObjTeam("WWranch", $OST_PLAYER);
   tsConvertObjTeam("WWwatertower", $OST_PLAYER);
   tsConvertObjTeam("WWtradingpost", $OST_PLAYER);
   tsConvertObjTeam("Civ1", $OST_PLAYER);
   tsConvertObjTeam("Civ2", $OST_PLAYER);
   tsConvertObjTeam("Civ3", $OST_PLAYER);
   tsConvertObjTeam("Civ4", $OST_PLAYER);
   
   // Start fire disaster
   %scenario = new SLScenario() {
      stack = true;
      disasterFireRate = "100 5";
   };
   
   // Update the scenario to include fires
   %scenario.disasterFireScenario = whiteWongCreateFire();
   LoadScenario(%scenario);
   
   // Force a disaster to occur
   %disaster = DisasterManager.OnCreateDisaster();
   
   // Add all buildings to disaster
   setBuildingAblaze("WWcabin1", %disaster);
   setBuildingAblaze("WWcabin2", %disaster);
   setBuildingAblaze("WWblacksmith", %disaster);
   setBuildingAblaze("WWsaloon", %disaster);
   setBuildingAblaze("WWstore", %disaster);
   setBuildingAblaze("WWranch", %disaster);
   setBuildingAblaze("WWtradingpost", %disaster);
}
//-WhiteWongFireHelpDialog Vertex-----------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Shuts off fire disasters and town asks for help with fire
////////////////////////////////////////////////////////////////////////////////
function WhiteWongFireHelpDialog::onEnter(%this)
{
    // End fire disasters
   %scenario = new SLScenario() {
      stack = true;
      disasterFireRate = "0 5";
   };
   %scenario.disasterFireScenario = whiteWongCreateFire();
   LoadScenario(%scenario);
   
   // Fire dialog
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_firehelpend", $TaskButton::Okay, "HeroCharacter", "id_wow_firehelpend");
   
   tsAddQuestToTracker(
      false,
      $WhiteWong::LogFireTask, 
      "", 
      $WhiteWong::LogFireTaskDesc, 
      $WhiteWong::LogFireTaskReward
      );
   tsReadyQuest($WhiteWong::LogFireTask);
   tsAddTaskToTracker($WhiteWong::LogFireTask, $WhiteWong::LogFireBrief, $WhiteWong::LogFireLong);
}
//-WhiteWongFireEnd Vertex------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Checks to see when the fire is over and completes this vertex
////////////////////////////////////////////////////////////////////////////////
function WhiteWongFireEnd::onUpdate(%this)
{
   Parent::onUpdate(%this);
   if(alertSvrGetAlertLocationCount($ALERT_FIRE) <= 0) {
      tsCompleteVertexes($WhiteWong::TaskName, "WhiteWongFireEnd");
   }
}
//-WhiteWongFireThanks Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Civilians thank hero and his group after a set amount of time elapses
////////////////////////////////////////////////////////////////////////////////
function WhiteWongFireThanks::onExit(%this)
{
   Parent::onExit(%this);
   
   for(%i = 1; %i <= 4; %i++) {
      if(isObject("Civ" @ %i)) {
         %obj = "Civ" @ %i;
         break;
      }
   }
   if(isObject(%obj)) {
      tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_firehelpthanks", $TaskButton::Okay, "Civ1");
      tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_firehelwelcome", $TaskButton::Okay, "HeroCharacter");
   }
   tsCompleteTaskInTracker($WhiteWong::LogFireTask, $WhiteWong::LogFireBrief);
   // Reward   
   tsModifyResource($Resource::Experience, 10, true);
   SendProductionToClient(HeroCharacter, "10 experience");
}
//-WhiteWongLose Vertex---------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Handles the lose condition (hero defeated, all units lost)
////////////////////////////////////////////////////////////////////////////////
function WhiteWongLose::onUpdate(%this)
{
   Parent::onUpdate(%this);
   if(HeroCharacter.isHealing && !tsCombatUnitsExist($OST_PLAYER)) {
      tsCompleteVertexes($WhiteWong::TaskName, "WhiteWongLose");
   }
}
//-WhiteWongLoseRestart Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Used to restart the scenario on lose
////////////////////////////////////////////////////////////////////////////////
function WhiteWongLoseRestart::onExit(%this)
{
   Parent::onExit(%this);
   tsRestartScenario();
}
//-WhiteWongBattleStart Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Responsible for the dialog between the fighting bandits and the player's team
////////////////////////////////////////////////////////////////////////////////
function WhiteWongBattleStart::onEnter(%this)
{
   Parent::onEnter(%this);
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_battle1", $TaskButton::Okay, "whitebandit1");
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_battle2", $TaskButton::Okay, "wongbandit1");
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_battle3", $TaskButton::Okay, "whitebandit3");
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_battle4", $TaskButton::Okay, "wongbandit3");
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_battlestart", $TaskButton::Okay, "HeroCharacter", "id_wow_battlestart");
}
//-WhiteWongBattleEnd Vertex----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Converts bandits to enemy team and monitors battle end
////////////////////////////////////////////////////////////////////////////////
function WhiteWongBattleEnd::onEnter(%this)
{
   tsConvertObjTeam("whitebandit1", $OST_BANDIT);
   tsConvertObjTeam("whitebandit2", $OST_BANDIT);
   tsConvertObjTeam("whitebandit3", $OST_BANDIT);
   tsConvertObjTeam("whitebandit4", $OST_BANDIT);
   tsConvertObjTeam("wongbandit1", $OST_BANDIT);
   tsConvertObjTeam("wongbandit2", $OST_BANDIT);
   tsConvertObjTeam("wongbandit3", $OST_BANDIT);
   tsConvertObjTeam("wongbandit4", $OST_BANDIT);
}
//-WhiteWongAppear--------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Spawns white and wong on the cliffs overlooking the battle site
////////////////////////////////////////////////////////////////////////////////
function WhiteWongBattleEnd::onExit(%this)
{
   Parent::onExit(%this);
   
   // Spawn white and wong
   %obj = tsCreateObj("MrWhiteData", "mrwhite", $OST_NEUTRAL, "-251 -191", "mrwhite");
   %obj.name = "Mr.White ";
   %obj1 = tsCreateObj("MrWongData", "mrwong", $OST_NEUTRAL, "-250 -162", "mrwong");
   tsNameObj(%obj1,"Mr.Wong ");
}
//-WhiteWongPanCamera Vertex----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Pans the camera to location between white and wong
////////////////////////////////////////////////////////////////////////////////
function WhiteWongPanCamera::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Disable input and pan camera
   tsEnableInput(false);
   tsMoveCameraToPosition("-250 -177", false);
}
function WhiteWongPanCamera::onUpdate(%this)
{
   Parent::onUpdate(%this);
   
   // Done when camera is done panning
   if(!tsIsCameraMoving()) {
      tsEnableInput(true);
      tsCompleteVertexes($WhiteWong::TaskName, "WhiteWongPanCamera");
   }
}
//-WhiteWongDialog Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Gives dialog between white and wong
////////////////////////////////////////////////////////////////////////////////
function WhiteWongDialog::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Dialog
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_whiteargue", $TaskButton::Okay, "mrwhite");
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_wongargue", $TaskButton::Okay, "mrwong");
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_endargument", $TaskButton::Okay, "mrwhite", "id_wow_endargument");
   tsSendTaskClickDlg("ID_TASK_CHOOSEWHITE", "id_wow_whitechoice", $TaskButton::Accept, "HeroCharacter", "id_wow_whitechoice");
   tsSendTaskClickDlg("ID_TASK_CHOOSEWONG", "id_wow_wongchoice", $TaskButton::Accept, "HeroCharacter", "id_wow_wongchoice");
}
function WhiteWongDialog::onExit(%this)
{
   Parent::onExit(%this);
   
   tsAddTaskToTracker($WhiteWong::LogMainTask, $WhiteWong::LogChooseBrief, $WhiteWong::LogChooseBrief);
   tsCompleteTaskInTracker($WhiteWong::LogMainTask, $WhiteWong::LogFindBrief);
}
//-WhiteWongChooseWhite Vertex-------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is marked complete if the player chooses white over wong.
// Possible to add a small cutscene after this vertex with another vertex or two
////////////////////////////////////////////////////////////////////////////////
function WhiteWongChooseWhite::onExit(%this)
{
   Parent::onExit(%this);
   
   if(%this.hasBeenCompleted()) {
      tsCompleteTaskInTracker($WhiteWong::LogMainTask, $WhiteWong::LogChooseBrief);
      tsCompleteVertexes($WhiteWong::TaskName, "WhiteWongChooseWong");
      tsDestroyTaskDlgIcon("id_wow_wongchoice", "HeroCharacter");
   }
}
//-WhiteWongChooseWong Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is marked complete if the player chooses wong over white.
// Possible to add a small cutscene after this vertex with another vertex or two
////////////////////////////////////////////////////////////////////////////////
function WhiteWongChooseWong::onExit(%this)
{
   Parent::onExit(%this);
   
   if(%this.hasBeenCompleted()) {
      tsCompleteTaskInTracker($WhiteWong::LogMainTask, $WhiteWong::LogChooseBrief);
      tsCompleteVertexes($WhiteWong::TaskName, "WhiteWongChooseWhite");
      tsDestroyTaskDlgIcon("id_wow_whitechoice", "HeroCharacter");
   }
}
function WhiteWongPanToPrisoner::onEnter(%this)
{
   Parent::onEnter(%this);
   tsMoveCameraToObject(whiteWongTaskPrisoner(), false);
}
function WhiteWongPanToPrisoner::onUpdate(%this)
{
   Parent::onUpdate(%this);   
   if(!tsIsCameraMoving()) {
      tsCompleteVertexes($WhiteWong::TaskName, "WhiteWongPanToPrisoner");
   }
}
function WhiteWongPrisonerDialog::onEnter(%this)
{
   Parent::onEnter(%this);
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_prisonerdialog", $TaskButton::Okay, whiteWongTaskPrisoner(), "id_wow_prisonerdialog");
}
function WhiteWongPanToEscapee::onEnter(%this)
{
   Parent::onEnter(%this);
   tsMoveCameraToObject(whiteWongTaskEscapee(), false);
}
function WhiteWongPanToEscapee::onUpdate(%this)
{
   Parent::onUpdate(%this);   
   if(!tsIsCameraMoving()) {
      tsCompleteVertexes($WhiteWong::TaskName, "WhiteWongPanToEscapee");
   }
}
function WhiteWongEscapeeDialog::onEnter(%this)
{
   Parent::onEnter(%this);
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_escapeedialog", $TaskButton::Okay, whiteWongTaskEscapee(), "id_wow_escapeedialog");
}
function WhiteWongEscape::onEnter(%this)
{
   Parent::onEnter(%this);
   %escapee = whiteWongTaskEscapee();
   if(%escapee $= "mrwhite") {
      %loc = "-97 211";
   }
   else {
      %loc = "-134 -145";
   }
   tsMoveObjToTarget(%escapee, %loc);
}
function WhiteWongEndDialogWait::onExit(%this)
{
   Parent::onExit(%this);
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_returntomain", $TaskButton::Okay, "HeroCharacter", "id_wow_returntomain");
}
function WhiteWongEscapeeFade::onUpdate(%this)
{
   Parent::onUpdate(%this);
   %escapee = whiteWongTaskEscapee();
   %cmpAI   = slgQueryInterface(%escapee, $CID_AI);
   if(!IsFollowingPath(%cmpAI)) {
      tsCompleteVertexes($WhiteWong::TaskName, "WhiteWongEscapeeFade");
      tsShowMesh(%escapee, false);
   }
}
//-WhiteWongReturnToMain--------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Takes the player back to the main campaign map
////////////////////////////////////////////////////////////////////////////////
function WhiteWongReturnToMainWait::onExit(%this)
{
   Parent::onExit(%this);
   %scenario = new SLScenario() {
      stack = false;
   };
    // Mark white as chosen
   if(WhiteWongChooseWhite.hasBeenCompleted()) {
      LoadScenario(%scenario, endWhiteWongTasks, chooseWhite);
   }
   // Mark wong as chosen
   else if(WhiteWongChooseWong.hasBeenCompleted()) {
      LoadScenario(%scenario, endWhiteWongTasks, chooseWong);
   }
   else {
      LoadScenario(%scenario, endWhiteWongTask);
   }
   
   // complete the white or wong satellite
   mapsatellite_whiteorwong.disabled = false;
}
//-WhiteWongPit Vertex----------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex controls the dynamite pit zone. Once entered, the dynamite object
// is spawned
////////////////////////////////////////////////////////////////////////////////
function WhiteWongPit::onMessage(%this, %msg, %params)
{
   Parent::onMessage(%this, %msg, %params);
   
   if(%this.hasBeenCompleted()) {
      // Send message
      %objs = tsGetObjListFromMsg(%msg, %params);
      %char = %objs.getRandomObject();
      if(%char.getName() $= "") {
         %char.setName("dynamitetrigger");
      }
      
      // Create dynamite
      %obj = tsCreateObj("Dynamite", "TNT", $OST_NEUTRAL, "-402 -272");
      
      // Damage objects
      %count = %objs.getSize();
      for(%i = 0; %i < %count; %i++) {
         %char = %objs.getID(%i);
         %char.health -= $WhiteWong::DynamiteDamage;
      }
   }
}
//-WhiteWongPitExplode Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for starting the particle effect on the dynamite
// This MUST happen after the dynamite prop exists on the client (which is why
// there is a slight delay)
////////////////////////////////////////////////////////////////////////////////
function WhiteWongPitExplode::onExit(%this)
{
   Parent::onExit(%this);
   
   // Destroy dynamite
   tsShowMesh("TNT", false);
   tsStartObjParticleFx("TNT", "dynamite");
   %count = ClientGroup.getCount();
   for(%i = 0; %i < %count; %i++) {
      %client = ClientGroup.getObject(%i);
      commandToClient(%client, 'PlaySFXSound', audio_dynamiteexplode);
   }
}
//-WhiteWongPitDestroy Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Responsible for destroying the dynamite prop and giving a response dialog
////////////////////////////////////////////////////////////////////////////////
function WhiteWongPitDestroy::onExit(%this)
{
   Parent::onExit(%this);
   
   // Response dialog and destruction
   tsRemoveObj("TNT");
   
   if(isObject(dynamitetrigger)) {
      %char = dynamitetrigger;
   }
   else {
      %char = HeroCharacter;
   }
   tsSendTaskPopUpDlg("ID_TASKNAME_WOW", "id_wow_tntpit", $TaskButton::Okay, %char);
}

// End WhiteWong.cs